/********************************************************************** 
**  Oracle_Mangler_Script.sql 
** 
**  Mangler for Primavera  Oracle DB 
** 
**  Modified 12-Jan-2017
** 
**  Description: Changes contents of string and memo fields in  
**  Primavera PMDB database. 
** 
**  Req:   
**  1. Run as 'admuser'
***********************************************************************/

DECLARE
TYPE chgblobtyp IS REF CURSOR;
blob_curr CHGBLOBTYP;

CURSOR TABLES IS
SELECT table_name table_name,
column_name column_name
FROM user_tab_columns
WHERE data_type = 'VARCHAR2'
AND column_name NOT IN ( 'GUID', 'TMPL_GUID', 'PASSWD', 'UDF_TYPE_NAME', 
                         'RPT_STATE', 'WINDOW_START', 'WINDOW_END', 
                         'DATABASE_VERSION', 'TABLE_NAME', 'SETTING_NAME',
                         'DASHBOARD_LAYOUT_DATA', 'FORMULA',
                         'RSRC_PRIVATE', 'VIEW_NAME', 'ISSUPERUSER',
                         'UDF_TYPE_LABEL')
AND table_name NOT IN ( 'ADMIN_CONFIG', 'BRE_REGISTRY', 'CURRTYPE',
                        'USESSION',
                        'REFRDEL', 'HQUERY', 'NEXTKEY', 'PUBUSER',
                        'QUERYLIB', 'PKXREF', 'USERCOL', 'USERSET',
                        'REITTYPE', 'M2_LOG', 'PROJPROP', 'RSRCPROP',
                        'LOCATION', 'PROJSHAR', 'VWPREFDATA', 'ATTR_MASTER' )
AND column_name NOT LIKE '%\_FLAG' ESCAPE '\'
AND column_name NOT LIKE '%\_TYPE' ESCAPE '\'
AND column_name NOT LIKE '%\_TYPE2' ESCAPE '\'
AND column_name NOT LIKE '%\_CHAR' ESCAPE '\'
AND column_name NOT LIKE '%\_KEY' ESCAPE '\'
AND column_name NOT LIKE '%\_SCOPE' ESCAPE '\'
AND column_name NOT LIKE '%\_LEVEL' ESCAPE '\'
AND column_name NOT LIKE '%\_ABBREV' ESCAPE '\'
AND column_name NOT LIKE 'STATUS%'
AND column_name NOT LIKE '%\_PRIVATE' ESCAPE '\'
AND column_name NOT LIKE '%_INTERVAL' ESCAPE '\'
UNION
SELECT 'ACTVTYPE' table_name,
'ACTV_CODE_TYPE' column_name
FROM dual
UNION
SELECT 'BASETYPE' table_name,
'BASE_TYPE' column_name
FROM dual
UNION
SELECT 'RISKTYPE' table_name,
'RISK_TYPE' column_name
FROM dual
UNION
SELECT 'MEMOTYPE' table_name,
'MEMO_TYPE' column_name
FROM dual
UNION
SELECT 'PCATTYPE' table_name,
'PROJ_CATG_TYPE' column_name
FROM dual
UNION
SELECT 'RCATTYPE' table_name,
'RSRC_CATG_TYPE' column_name
FROM dual
ORDER BY 1, 2;

CURSOR blobs IS
SELECT table_name, column_name, data_type
FROM user_tab_columns
WHERE data_type IN ( 'CLOB', 'BLOB' )
AND column_name NOT IN ( 'PROP_VALUE' )
AND column_name NOT LIKE '%\_DATA' ESCAPE '\'
AND table_name NOT IN ( 'USERSET', 'VWPREFDATA' );

v_version VARCHAR2(17);
v_len_col VARCHAR2(255);
v_col_len NUMBER;
v_min_col_len NUMBER;
v_min_col_name VARCHAR2(255);
v_key_cnt NUMBER;
v_key_name VARCHAR2(255);
v_pk_column VARCHAR2(255);
v_pk_column2 VARCHAR2(255);
v_upd VARCHAR2(1000);
v_prefix VARCHAR2(2);
v_pad_raw RAW(4000);
v_pad_char VARCHAR2(4000);
v_sql VARCHAR2(4000);
v_blob BLOB;
v_clob CLOB;
v_lob_len NUMBER;
v_pk1 NUMBER;
v_pk2 NUMBER;
v_remain NUMBER;
v_fill_size NUMBER;
v_offset NUMBER;
v_process BOOLEAN;
vtable_name VARCHAR2(255);
v_datatype VARCHAR2(255);
v_pk2v VARCHAR2(255);

PROCEDURE Writelog (msg VARCHAR2)
IS
PRAGMA autonomous_transaction;
BEGIN
  EXECUTE IMMEDIATE 'insert into m2_log (ts,message) values (:ts,:msg)'
  USING SYSDATE, msg;
  
  COMMIT;
END;

FUNCTION Has_nextkey (pkey_name VARCHAR2)
RETURN BOOLEAN
IS
dummy NUMBER;
BEGIN
  SELECT key_seq_num
  INTO dummy
  FROM nextkey
  WHERE key_name = Lower(pkey_name);
  
  RETURN TRUE;
EXCEPTION
  WHEN no_data_found THEN
  RETURN FALSE;
END;

FUNCTION Is_primavera_table (tab VARCHAR2,
                             keycolumn VARCHAR2,
                             keycnt NUMBER)
RETURN BOOLEAN
IS
CURSOR parent IS
SELECT rc.table_name
|| '_'
|| cc.column_name key_name
FROM user_constraints uc,
user_constraints rc,
user_cons_columns cc
WHERE uc.table_name = tab
AND uc.constraint_type = 'R'
AND uc.owner = rc.owner
AND rc.constraint_name = uc.r_constraint_name
AND cc.owner = rc.owner
AND cc.constraint_name = rc.constraint_name
AND cc.position = 1;
nk VARCHAR2(100);
BEGIN
      -- Exception to the rule
  IF tab = 'PREFER' THEN
    RETURN TRUE;
  END IF;
  
  nk := tab || '_' || keycolumn;
  
  IF Has_nextkey(tab || '_' || keycolumn) THEN
    RETURN TRUE;
  END IF;
  
  FOR parent_rec IN parent LOOP
    IF Has_nextkey(parent_rec.key_name) THEN
      RETURN TRUE;
    END IF;
  END LOOP;
  
  RETURN FALSE;
END;

BEGIN /*Main Process*/
  EXECUTE IMMEDIATE 'select version from v$instance' INTO v_version;
  
  v_version := Substr(v_version, 1, Instr(v_version, '.') - 1);
  
  IF v_version = '8' THEN
    v_len_col := 'DATA_LENGTH';
  ELSE
    v_len_col := 'CHAR_LENGTH';
  END IF;
  
  BEGIN
    EXECUTE IMMEDIATE 'drop table m2_log';
  EXCEPTION
    WHEN OTHERS THEN
    NULL;
  END;
  
  EXECUTE IMMEDIATE 'create table m2_log (ts date,message varchar2(1000))';
  
  v_min_col_len := 4001;
  
  --Process fields Begin
  FOR tab_rec IN TABLES LOOP
    v_process := TRUE;
    
    IF tab_rec.table_name != 'GCHANGE'
      AND tab_rec.table_name != 'RLFOLIO' THEN
      v_key_name := 'PK_' || tab_rec.table_name;
    ELSE
      v_key_name := 'PK_' || tab_rec.table_name || '_ID';
    END IF;
    
    SELECT MAX(position)
    INTO v_key_cnt
    FROM user_cons_columns
    WHERE constraint_name = v_key_name;
    
    IF v_key_cnt > 0 THEN
      SELECT column_name
      INTO v_pk_column
      FROM user_cons_columns
      WHERE constraint_name = v_key_name
      AND position = 1;
    ELSE
      v_process := FALSE;
    END IF;
    
    IF v_process
      AND Is_primavera_table(tab_rec.table_name, v_pk_column, v_key_cnt)
      THEN
      Writelog('Mangling: '|| tab_rec.table_name || '.' || tab_rec.column_name);
      
      BEGIN
        EXECUTE IMMEDIATE 'select '|| v_len_col ||' from user_tab_columns where table_name = :tab and column_name = :col'
        INTO v_col_len
        USING tab_rec.table_name, tab_rec.column_name;
      EXCEPTION
        WHEN OTHERS THEN
        Writelog('Exception: v_pd = '|| v_upd);
      END;
      
      IF v_col_len < v_min_col_len THEN
        v_min_col_len := v_col_len;
        v_min_col_name := tab_rec.table_name || '.' || tab_rec.column_name;
      END IF;
      
      IF tab_rec.column_name = 'TASK_CODE' THEN
        v_prefix := 'A';
      ELSE
        v_prefix := Substr(tab_rec.table_name, 1, 1);
      END IF;
      
	  --process alternate updates for different fields
      IF tab_rec.column_name = 'USER_NAME' THEN
        v_upd := 'update users set user_name = ''U'' || to_char(user_id)';
        EXECUTE IMMEDIATE v_upd;
      ELSIF tab_rec.column_name LIKE '%SHORT_NAME' THEN
        v_upd := 'update '|| tab_rec.table_name || ' set '|| tab_rec.column_name || ' = '''|| v_prefix ||''' || TO_CHAR('|| v_pk_column || ')';
        EXECUTE IMMEDIATE v_upd;
      ELSE
        BEGIN
		  -- First: Update with padding
          v_upd := 'update '|| tab_rec.table_name || ' set '|| tab_rec.column_name || ' = RPAD('''|| v_prefix ||''' || TO_CHAR('|| v_pk_column ||')|| '' '' , LENGTH('|| tab_rec.column_name || '),''xX'')' || ' WHERE LENGTH('''|| v_prefix ||''' || TO_CHAR('|| v_pk_column ||')|| '' '') < LENGTH('|| tab_rec.column_name || ')';
          EXECUTE IMMEDIATE v_upd;
		  -- Second: Update without padding
          v_upd := 'update '|| tab_rec.table_name || ' set '|| tab_rec.column_name || ' = '''|| v_prefix ||''' || TO_CHAR('|| v_pk_column || ')' || ' WHERE LENGTH('''|| v_prefix ||''' || TO_CHAR('|| v_pk_column ||')|| '' '') >= LENGTH('|| tab_rec.column_name || ')';
          EXECUTE IMMEDIATE v_upd;
          COMMIT;
        EXCEPTION
          WHEN OTHERS THEN
          Writelog('Exception: v_pd = '|| v_upd);
        END;
      END IF;
    END IF;
  END LOOP; --Process fields End
  
  dbms_output.Put_line ('Minimum column length is '|| To_char(v_min_col_len) || ' for '|| v_min_col_name);
  
  UPDATE users
  SET passwd = NULL;
  
  --Process memo fields Begin
  FOR tab_rec IN blobs LOOP
    v_process := TRUE;
    
    IF tab_rec.table_name != 'GCHANGE'
      AND tab_rec.table_name != 'RLFOLIO' THEN
      v_key_name := 'PK_' || tab_rec.table_name;
    ELSE
      v_key_name := 'PK_' || tab_rec.table_name || '_ID';
    END IF;
    
    SELECT MAX(position)
    INTO v_key_cnt
    FROM user_cons_columns
    WHERE constraint_name = v_key_name;
    
    IF v_key_cnt > 0 THEN
      SELECT column_name
      INTO v_pk_column
      FROM user_cons_columns
      WHERE constraint_name = v_key_name
      AND position = 1;
    ELSE
      v_process := FALSE;
    END IF;
    
    IF v_process
      AND Is_primavera_table(tab_rec.table_name, v_pk_column, v_key_cnt)
      THEN
      Writelog('Mangling blob: '|| tab_rec.table_name || '.' || tab_rec.column_name);
      
      IF v_key_cnt = 2 THEN
        SELECT column_name, table_name
        INTO v_pk_column2, vtable_name
        FROM user_cons_columns
        WHERE constraint_name = v_key_name
        AND position = 2;
        
        SELECT data_type
        INTO v_datatype
        FROM user_tab_columns
        WHERE column_name = v_pk_column2
        AND table_name = vtable_name;
      END IF;
      
      IF v_key_cnt = 2 THEN
        v_sql := 'select '|| v_pk_column || ',' || v_pk_column2;
      ELSE
        v_sql := 'select '|| v_pk_column;
      END IF;
      
      v_sql := v_sql || ',' || tab_rec.column_name || ', dbms_lob.getlength('|| tab_rec.column_name || ') from '|| tab_rec.table_name || ' where '|| tab_rec.column_name || ' is not null '|| ' for update of '|| tab_rec.column_name;
      
      OPEN blob_curr FOR v_sql;
      LOOP
        IF tab_rec.data_type = 'BLOB' THEN
          IF v_key_cnt = 2
            AND v_datatype = 'NUMBER' THEN
            FETCH blob_curr INTO v_pk1, v_pk2, v_blob, v_lob_len;
          ELSIF v_key_cnt = 2
            AND v_datatype = 'VARCHAR2' THEN
            FETCH blob_curr INTO v_pk1, v_pk2v, v_blob, v_lob_len;
          ELSIF v_key_cnt = 1 THEN
            FETCH blob_curr INTO v_pk1, v_blob, v_lob_len;
          END IF;
        ELSE
          IF v_key_cnt = 2 THEN
            FETCH blob_curr INTO v_pk1, v_pk2, v_clob, v_lob_len;
          ELSE
            FETCH blob_curr INTO v_pk1, v_clob, v_lob_len;
          END IF;
        END IF;
        
        v_remain := v_lob_len;
        v_offset := 1;
        
        WHILE v_remain > 0 LOOP --1  
          BEGIN --2
            IF v_remain < 4000 THEN
              v_fill_size := v_remain + MOD(v_remain, 2);
            ELSE
              v_fill_size := 4000;
            END IF;
            
            IF tab_rec.data_type = 'BLOB' THEN
              v_pad_raw := Hextoraw(Rpad('58', v_fill_size, '58')) || Hextoraw(Rpad('58', v_fill_size, '58'));
              dbms_lob.WRITE(v_blob, v_fill_size, v_offset, v_pad_raw);
            ELSE
              v_pad_char := Rpad ('X', v_fill_size, 'X');
              dbms_lob.WRITE(v_clob, v_fill_size, v_offset, v_pad_char);
            END IF;
            
            v_remain := v_remain - 4000;
            v_offset := v_offset + 4000;
          EXCEPTION
            WHEN OTHERS THEN
            Writelog('Exception in : '|| tab_rec.column_name || 'Error code message:'|| SQLERRM);
            Writelog('Exception: v_fill_size = '|| v_fill_size);
            Writelog('Exception: v_offset = '|| v_offset);
            Writelog('Exception: v_pk1 = '|| v_pk1);
          END; --2
        END LOOP; --1
        EXIT WHEN blob_curr%notfound;
      END LOOP;
    END IF;
  END LOOP;
  COMMIT;
END;

/ 